FUNCTION sppair,spa,spb,SIGMA=sigma,WIDTH=width,DIAG=diag

; 12 Nov 05 if error near zero, return error in first spectrum
; 19 Dec 04 repaired bug with weighting
; 15 Dec 04 reduce std dev to uncertainty in mean where both nods good
; 21 Mar 04 either reject or use data, do not use continuum, return uncertainty
;  8 Mar 04 created
;
; given two spectral data arrays, combine them and eliminate spikes
; algorithm:
;   generate continuum spectra for each input spectrum
;   generate difference spectra for each = abs(continuum-spectrum)
;   generate pass spectrum:
;   (1) at each wavelength, choose min of the two diff spectra
;   (2) replace small pass values by raising to median or mean of pass array
;   generate a weight array by comparing differences to pass array
;       w=0 if diff > pass*sigma, otherwise w=1
;       if both weights=0 for a given wavelength element, reset to 1
;   combine spectra by averaging
;   compute uncertainty as though all data were used
; INPUT
;   spa,spb - spectral data arrays
;   SIGMA   - number of times pass before data replaced
;   WIDTH   - size of box for median
;   DIAG    - keyword to turn diagnostic plotting on
; OUTPUT
;   spike-corrected and averaged spectrum

diagmode=2

; check keywords

if (keyword_set(sigma) eq 0) then sigma=5.0
if (keyword_set(width) eq 0) then width=5
if (keyword_set(diag) eq 0)  then diagflag=0 else diagflag=diagmode

lcol=0 & fcol=1 & ecol=2 & ocol=3

; check that spectral arrays are the same size
; and initialize new spectral data array newspec

len=n_elements(spa[lcol,*])
if (len ne n_elements(spb[lcol,*])) then begin
  print,"Error.  Spectral sizes do not match."
  stop
endif
newspec=spa

; load wavelength, flux, continuum, diff spectra, pass spectrum
; continuum spectra are media of flux spectra

lam  =reform(spa[lcol,*])
fluxa=reform(spa[fcol,*])
fluxb=reform(spb[fcol,*])
erra =reform(spa[ecol,*])
cona =median(fluxa,width)
conb =median(fluxb,width)
diffa=abs(fluxa-cona)
diffb=abs(fluxb-conb)
pass =fltarr(len)

for i=0,len-1 do pass[i]=min([diffa[i],diffb[i]]) ; minimum difference

if (diagflag eq 1) then begin
  spplot,spa
  spplot,spb,/over
endif
if (diagflag eq 2) then begin
   plot,lam,diffa
   oplot,lam,diffb
endif

; raise low values of pass to smaller of median or mean of pass array 

;minpass=max([median(pass),mean(pass)])
minpass=median(pass)
bad_idx=where(pass lt minpass)
if (diagflag gt 0) then $
  print,'median ',median(pass),' mean ',mean(pass),' minpass ',minpass
if (max(bad_idx) gt -1) then pass[bad_idx] = minpass
if (diagflag eq 2) then oplot,lam,sigma*pass,th=2

; set weight arrays wt_a and wt_b
; default is 1.0
; if both wt_a and wt_b for a given wavelength element=0, reset to 1
;   this shouldn't happen, but it would indicate the pixel is hopeless

wt_a=fltarr(len)+1.0
wt_b=wt_a

bad_idx=where(diffa gt sigma*pass)
if (max(bad_idx) gt -1) then begin
  wt_a[bad_idx]=0.0
  if (diagflag eq 1) then oplot,lam[bad_idx],fluxa[bad_idx],psym=4
  if (diagflag eq 2) then oplot,lam[bad_idx],diffa[bad_idx],psym=4
endif
bad_idx=where(diffb gt sigma*pass)
if (max(bad_idx) gt -1) then begin
  wt_b[bad_idx]=0.0
  if (diagflag eq 1) then oplot,lam[bad_idx],fluxa[bad_idx],psym=4
  if (diagflag eq 2) then oplot,lam[bad_idx],diffb[bad_idx],psym=4
endif
bad_idx=where(wt_a eq 0.0 and wt_b eq 0.0)
if (max(bad_idx) gt -1) then begin
  print,'There are ',n_elements(bad_idx),' hopeless pixels'
  wt_a[bad_idx] =1.0
  wt_b[bad_idx] =1.0
endif

; combine spectra by weighted averaging
; generate new uncertainties
; divide uncertainties by sqrt(2) where both nods good

newspec[1,*]=(fluxa*wt_a+fluxb*wt_b)/(wt_a+wt_b)
for i=0,len-1 do newspec[ecol,i]=stddev([fluxa[i],fluxb[i]])
idx=where(newspec[fcol,*] ne fluxa and newspec[fcol,*] ne fluxb)
if (max(idx) gt -1) then newspec[ecol,idx]=newspec[ecol,idx]/sqrt(2.0)

; if error < 0.1% flux, then replace with error from first spectrum

idx=where(newspec[ecol,*] lt 0.001*abs(newspec[fcol,*]))
if (max(idx) gt -1) then newspec[ecol,idx]=erra[idx]

if (diagflag eq 1) then spplot,newspec,th=2,/over

return,newspec

END
