FUNCTION spex,spdata,spcont,l0,l1,lam_center,ORDER=order,MODE=mode,FLUX=flux,RATIO=ratio,DIAG=diag

; 12 Sep 05 added mode keyword as an alternative way of invoking flux or ratio 
;           also allowing spcont=scalar zero now (for no background)
;  6 Sep 05 only sorting if order not set, sorting dellam vector properly now
; 31 Aug 05 renamed as spex, added error to lam_center, /ratio keyword
; 30 Aug 05 added lam_center parameter to find central wavelength
; 29 Mar 05 added /flux keyword
;  5 Mar 05 modified to use internal uncertainty when available
; 15 Oct 04 created as spexw
;
; given a spectral data array and corresponding continuum
; determine equivalent width of feature from l0 to l1 (or flux or flux ratio)
; both spectra data arrays must be on the same wavelength grid
;
; assumed that flux is in Jy
; flux and error units converted to W m^-2 um^-1
;
; INPUT
;   spdata     - spectral data array containing the original spectrum
;   spcont     - spectral data array containing the continuum estimate
;              - note, if spcont = scalar 0, then a vector of zeroes is used
;   lo,l1      - starting and ending wavelength to find equivalent width
;   order      - keyword to limit analysis to one order
;   mode       - optional parameter to set flux (1) or ratio (2) keywords
;   flux       - keyword to return equiv. flux instead of equiv. width (mode=1)
;   ratio      - keyword to return ratio of eq flux to continuum (mode=2)
;   lam_center - parameter used to return central wavelength  and error
;                defined as wavelength with half the flux on either side
;   diag       - optional keyword to turn on diagnostic mode
; OUTPUT - returns equivalent width and uncertainty as a two-element vector

lcol=0 & fcol=1 & ecol=2 & ocol=3

; check keywords - mode supersedes flux or ratio keywords

if (keyword_set(mode) eq 0) then begin
  mode=0
  if (keyword_set(flux) eq 1) then mode=1
  if (keyword_set(ratio) eq 1) then mode=2
endif 

if (n_elements(order) gt 0) then ordflag=1 else ordflag=0

; if spcont = scalar 0, then replaced with vector fo zeroes

if (n_elements(spcont) eq 1 and total(spcont) eq 0) then begin
  spcont=sp & spcont[fcol]=0 & spcont[ecol]=0
endif

; check size of arrays

if (n_elements(spdata[lcol,*]) ne n_elements(spcont[lcol,*])) then begin
  print,'Error in spexw.  Spectrum and continuum have different lengths.'
endif

; check order keyword and either load relevant data into spo and spc or sort

spo=spdata & spc=spcont 
if (ordflag eq 1) then begin
  idx=where(spdata[ocol,*] eq order)
  if (max(idx) gt -1) then begin
    spo=spdata[*,idx]
    spc=spcont[*,idx]
  endif else begin
    print,'Warning in spexw.  No data in order, using full spectrum.'
    ordflag=0
  endelse
endif 

if (ordflag eq 0) then begin
  idx=sort(spo[lcol,*])
  spo=spo[*,idx]
  spc=spc[*,idx]
endif

; set up dellam vector - contains delta lambda for each pixel

dellam=abs(reform(shift(spo[lcol,*],1)-spo[lcol,*]))
dellam[0]=dellam[1]                        ; must replace first pixel 

; convert spectral data from F_nu units to F_lambda units 

spo[fcol,*] = spo[fcol,*] * 3e-12 / (spo[lcol,*]^2)
spo[ecol,*] = spo[ecol,*] * 3e-12 / (spo[lcol,*]^2)
spc[fcol,*] = spc[fcol,*] * 3e-12 / (spc[lcol,*]^2)

; find wavelengths for summing equivalent width

idx=where(spo[lcol,*] ge l0 and spo[lcol,*] le l1)

if (max(idx) gt -1) then begin

; NOW finding S/N ratio from error column
; had been using smoothing flux in this range 
; also must find mean wavelength element
; use these to find uncertainty in equivalent width

;  spsmoo=spsmooth(spsmooth(spo[*,idx],7),7)
;  snr=mean(spo[fcol,idx])/stddev(spsmoo[fcol,*]-spo[fcol,idx])
  snr=mean(spo[fcol,idx]/spo[ecol,idx])
  wavel=mean(dellam)
  sigw=sqrt(2.0*n_elements(idx))*wavel/snr
  sigf=sqrt(total(spo[ecol,idx]^2*dellam[idx]^2))

; find equivalent width, eq flux, integrated flux of continuum

  eqw=total( (1.0-(spo[fcol,idx]/spc[fcol,idx])) * dellam[idx] )
  eqf=total((spo[fcol,idx]-spc[fcol,idx])*dellam[idx])
  eqc=total(spc[fcol,idx]*dellam[idx])

; return value depends on mode, 0 - eq width, 1 - eq flux, 2 - flux ratio

  case mode of
    0 : retval=[eqw,sigw]
    1 : retval=[eqf,sigf]
    2 : retval=[eqf/eqc,sigf/eqc] ; divide flux, error by integrated continuum
  endcase

; find lam_center - central wavelength of feature
; to find error, find wavelength corresponding to half flux +/- sigma and avg
; do this by looping

  sumf=(spo[fcol,idx]-spc[fcol,idx])*dellam[idx]
  for j=n_elements(idx)-1,0,-1 do sumf[j]=total(sumf[0:j])
  sumf=sumf/eqf
  lam_c=fltarr(3)
  spot=[0.5-sigf/eqf,0.5,0.5+sigf/eqf]
  for i=0,2 do begin
    i0=max(where(sumf le spot[i])) & if (i0 le -1) then i0=0
    i1=min(where(sumf ge spot[i])) & if (i1 le -1) then i1=n_elements(idx)-1
    lam0=spo[lcol,idx[i0]] 
    lam1=spo[lcol,idx[i1]]
    if (lam0 eq lam1) then lam_c[i]=lam0 else begin
      lam_c[i] = lam0 + (lam1-lam0)*(spot[i]-sumf[i0]) / (sumf[i1]-sumf[i0])
      if (keyword_set(diag) eq 1) then print,i0,lam0,sumf[i0]
      if (keyword_set(diag) eq 1) then print,i1,lam1,sumf[i1]
    endelse
  endfor
  lam_center=[lam_c[1],0.5*abs(lam_c[2]-lam_c[0])]

endif else begin

  print,'Warning in spexw.  No data in range, returning zeroes.'
  retval=[0.0,0.0]
  lam_center=[0.0,0.0]

endelse

return,retval
END
